// File: applib.cp
//
// Description: Application specific code for the Macintosh
//              version of the Generic GUI toolkit. Many of
//              the "tricks" used here for interfacing C++
//              with the Mac toolbox are modeled after the
//              excellent example programs provided with the
//              MPW C++ compiler. 
//              
//
// Copyright 1992 by Mark Watson Associates
//
//       No binary rights reserved: this software library may
//       be used in compiled form without restrictions.
//       All source rights reserved: Source code to the GUI
//       library can not be distributed (on bulletin boards, 
//       or as part of shareware or commercial products)
//       without written permission.
//
//       This software is provided "as is". The user accepts
//       all responsibility for its use.
//


#include <Fonts.h>
#include <Controls.h>
#include <Windows.h>
#include <Menus.h>
#include <TextEdit.h>
#include <Dialogs.h>
#include <Desk.h>
#include <Scrap.h>
#include <ToolUtils.h>
#include <Memory.h>
#include <SegLoad.h>
#include <Files.h>
#include <Lists.h>
#include <OSUtils.h>
#include <StandardFile.h>
#include <Traps.h>
#include <pascal.h>

#include "applib.h"
    
// Define HiWrd and LoWrd macros for efficiency
// (needed for MPW, defined in Symantec)
//#define HiWrd(aLong)  (((aLong) >> 16) & 0xFFFF)
//#define LoWrd(aLong)  ((aLong) & 0xFFFF)


// Constants defined in MPW C++ examples:
const short kOsEvent = app4Evt;             // event used
                                            // by MultiFinder
const short kSuspendResumeMessage = 0x01;   // high byte of 
                                            // suspend/resume
                                            // event message
const short kResumeMask = 0x01;             // bit of message
                                            // field for resume
                                            // vs. suspend
const short kMouseMovedMessage = 0xFA;      // high byte of
                                            // mouse-moved
                                            // event message

// I define these as static local instead of
// in the Application class to avoid extra
// Mac specific include files in 'applib'h'.
// This speeds up compilation of application
// programs considerably.

static Boolean          fHaveWaitNextEvent;
static Boolean          fDone;
static RgnHandle        fMouseRgn;
static EventRecord      fTheEvent;
static WindowPtr        fWhichWindow;

TAppWindow *current_window;

// Utility from Aple MPW C++ Example Programs:
TrapAvailable(short tNumber,TrapType tType)
{
    return NGetTrapAddress(tNumber, tType) !=
           GetTrapAddress(_Unimplemented);
}



////////////////////////////////////////////////////////////////
//                                                            //
//                    Application Method Definitions:         //
//                                                            //
////////////////////////////////////////////////////////////////


Application::Application(void)
{
    SysEnvRec envRec;
    
    // Initialize the Mac Toolbox components
    InitGraf((Ptr) &thePort);
    InitFonts();
    InitWindows();
    InitMenus();
    TEInit();
    InitDialogs((ResumeProcPtr) nil);
    InitCursor();

    (void) SysEnvirons(curSysEnvVers, &envRec);

    // Are we running on a 128K ROM machine or better???
    if (envRec.machineType < 0) {
        Warning("This machine does not have 128K ROMs!");
        exit(1);
    }

    // Expand the heap so new code segments load at the top
    MaxApplZone();

    // check to see if WaitNextEvent is implemented
    fHaveWaitNextEvent = TrapAvailable(_WaitNextEvent, ToolTrap);

    // initialize our class variables
    fWindow = nil;
    fDone = false;
    fMouseRgn = nil;
    fWhichWindow = nil;
    
    Handle  menuBar;
    menuBar = GetNewMBar(rMenuBar);
    SetMenuBar(menuBar);
    DisposHandle(menuBar);
    
//  9/11/94: ombs under Symantec:
    AddResMenu(GetMHandle(mApple), 'DRVR');
    DrawMenuBar();
    fMouseRgn = NewRgn();
    
    // Make a single document window for this application:
    fWindow = new TAppWindow(rDocWindow);

}

//
//                       Handle events:
//


void Application::EventLoop(void)
{
    int gotEvent;
    EventRecord tEvt;
    short partCode;
    unsigned char evType;
    long mResult;
    WindowPtr tWind;
    char key;
    
    current_window->idle_proc(); // do idle work

    while (fDone == false) {
        // always set up fWhichWindow before doing anything
        fWhichWindow = FrontWindow();
        // make sure we always draw into correct window
        SetPort(fWhichWindow);

        current_window->idle_proc();  // application idle
                                      // work callback
        
        if (fHaveWaitNextEvent) {
            gotEvent = WaitNextEvent(everyEvent, &tEvt,
                                     (unsigned long)0,
                                     fMouseRgn);
        } else {
            SystemTask();
            gotEvent = GetNextEvent(everyEvent, &tEvt);
        }
        fTheEvent = tEvt;

        // make sure we got a real event
        if ( gotEvent ) {
            switch (fTheEvent.what) {
                case mouseDown :
                    // As per Apple program examples,
                    // make sure that we
                    // dereference object field references:
                    partCode = FindWindow(fTheEvent.where,
                                          &tWind);
                    fWhichWindow = tWind;
                    tEvt = fTheEvent;
                    switch (partCode) {
                        case inSysWindow :
                            break;
                        case inMenuBar :
                            mResult = MenuSelect(tEvt.where);
                            if (mResult != 0)
                              DoMenuCommand((short)HiWrd(mResult),
                                            (short)LoWrd(mResult));
                            break;
                        case inGoAway :
                            break;
                        case inDrag :
                            DragWindow(fWhichWindow,
                                       fTheEvent.where,
                                       &screenBits.bounds);
                            break;
                        case inGrow :
                            break;
                        case inZoomIn :
                        case inZoomOut :
                            break;
                        case inContent :
                            // If window is not in front,
                            // make it so:
                            if ( fWhichWindow != FrontWindow() )
                              SelectWindow(fWhichWindow);
                            else if (fWindow != nil)
                              fWindow->DoContent(&tEvt);                    
                            break;
                    }
                    break;
                case mouseUp :
                    break;
                case keyDown :
                case autoKey :
                    key = (char)(fTheEvent.message & charCodeMask);
                    if ((fTheEvent.modifiers & cmdKey) &&
                        (fTheEvent.what == keyDown)) {
                        mResult = MenuKey(key);
                        if (mResult != 0) {
                            DoMenuCommand((short)HiWrd(mResult),
                                          (short)LoWrd(mResult));
                            return;
                        }
                    }
                    if (fWindow != nil) {
                        // no document key handling used...
                    }
                    break;
                case updateEvt :
                    // The event record contains window ptr
                    fWhichWindow = (WindowPtr) fTheEvent.message;
                    SetPort(fWhichWindow);
                    if (fWindow != nil)
                      fWindow->DoUpdate();
                    break;
                case diskEvt :
                    break;
                case activateEvt :
                    // The event record contains window ptr
                    fWhichWindow = (WindowPtr) fTheEvent.message;
                    SetPort(fWhichWindow);
                    break;
                case kOsEvent :
                    // MultiFinder check as per Apple
                    // example programs:
                    evType = (unsigned char)
                        (fTheEvent.message >> 24) & 0x00ff;
                    switch (evType) // High byte of message
                                    // is type of event
                    {
                        case kMouseMovedMessage :
                            // application idle work:
                            current_window->idle_proc(); 
                            break;
                        case kSuspendResumeMessage :
                            break;
                    }
                    break;
                default :
                    break;
              } // end switch (fTheEvent.what)
          }
      }
}

//
//                          Handle menu commands:
//

void Application::DoMenuCommand(short menuID, short menuItem)
{
    short       itemHit;
    Str255      daName;
    short       daRefNum;
    WindowPtr   window;
    TAppWindow* fAppCurDoc = (TAppWindow*) fWindow;

    window = FrontWindow();
    switch ( menuID )
      {
        case mApple:
            switch ( menuItem ) {
                case iAbout: 
                    // "About" box:
                    itemHit = Alert(rAboutAlert, nil);
                    break;
                default: 
                    GetItem(GetMHandle(mApple),
                            menuItem, daName);
                    daRefNum = OpenDeskAcc(daName);
                    break;
            }
            break;
        case mFile:
            fDone = true;
            break;
        case mApp:
            // user supplied menu action function:
            fAppCurDoc->do_menu_action(menuItem);
            break;
      }
    HiliteMenu(0);      // unhighlight menu 
}


///////////////////////////////////////////////////////////////
//                                                           //
//                        TAppWindow Method Definitions:     //
//                                                           //
///////////////////////////////////////////////////////////////
    
TAppWindow::TAppWindow(short resID)
{   
    SetPort(fDocWindow);
    ShowWindow(fDocWindow);     // Make sure the window
                                // is visible
    bottom_text_clip = 0; // flag to indicate no scrolling
                          // text area is set up
    in_scrolling_text_mode = 0;
    redraw_both_text_and_graphics = 0;
}


TAppWindow::~TAppWindow(void)
{
    DisposeWindow(fDocWindow);
}

void TAppWindow::draw_update_contents(void)
{
    if (in_scrolling_text_mode == 0 ||
        redraw_both_text_and_graphics == 1)
    {
        update_display();  // Defined in user code
    }
    if (bottom_text_clip > 0 &&
        (in_scrolling_text_mode == 1 ||
         redraw_both_text_and_graphics == 1))
    {
            
        Rect r;         
        SetRect(&r,left_text_clip,top_text_clip-2,
                right_text_clip,bottom_text_clip+2);
        EraseRect(&r);

        Rect saveVis= fDocWindow->portRect;
        SetRect(&r,left_text_clip+2,top_text_clip-2,
                right_text_clip-2,bottom_text_clip+2);
        ClipRect(&r);
            
        // Refresh any user scrolling text:
        int next = current_ring_buffer_start;
        for (int i=0; i<number_of_saved_lines; i++)
        {
            next--;
            if (next < 0)  next = number_of_saved_lines - 1;
            plot_string(left_text_clip + 3,
                        bottom_text_clip -i*(string_height(" ")),
                        saved_text[next]);
        }
        ClipRect(&saveVis);
        if (redraw_both_text_and_graphics == 1)
        {
            r.bottom += 2; r.top -= 2;
            FrameRect(&r);
        }
    }
}

void TAppWindow::DoUpdate(void)
{
    BeginUpdate(fDocWindow);  // this sets up the visRgn 
    if (EmptyRgn(fDocWindow->visRgn) == 0)
    {
        draw_update_contents();
    }
    EndUpdate(fDocWindow);
}


static int iabs(int val)
{
    if (val > 0)   return val;
    return -val;
}


void TAppWindow::DoContent(EventRecord* /*theEvent*/)
{
    Point where; GetMouse(&where); /* = theEvent->where;  */
    mouse_down(where.h, where.v);

    Point p1;
    while (Button()) {
        GetMouse(&p1);
        mouse_move(p1.h, p1.v);
    }
    GetMouse(&p1);
    mouse_up(p1.h, p1.v);
}

void TAppWindow::plot_line(int x1, int y1, int x2, int y2)
{
    SetPort(fDocWindow);
    MoveTo(x1, y1);
    LineTo(x2, y2);
    in_scrolling_text_mode = 0;
}

void TAppWindow::plot_string(int x1, int y1, char *cp)
{
    SetPort(fDocWindow);
    TextMode(srcCopy);
    MoveTo(x1, y1);
    unsigned char *pascal_cp = CtoPstr(cp); // copies in place
    DrawString(pascal_cp);
    in_scrolling_text_mode = 0;
}

void TAppWindow::plot_rect(int top, int right,
                           int bottom, int left)
{
    Rect r;
    SetRect(&r, left, top, right, bottom);
    FrameRect(&r);
    in_scrolling_text_mode = 0;
}

void TAppWindow::erase_rect(int top, int right,
                            int bottom, int left)
{
    Rect r;
    SetRect(&r, left, top, right, bottom);
    EraseRect(&r);
    in_scrolling_text_mode = 0;
}

void TAppWindow::plot_oval(int top, int right,
                           int bottom, int left)
{
    Rect r;
    SetRect(&r, left, top, right, bottom);
    FrameOval(&r);
    in_scrolling_text_mode = 0;
}


int  TAppWindow::string_height(char *)
{
    FontInfo fi;
    GetFontInfo(&fi);
    return (fi.ascent + fi.descent + fi.leading);
}


int  TAppWindow::string_width(char *str)
{
    FontInfo fi;
    GetFontInfo(&fi);
    return (fi.widMax * strlen(str) / 2) + 2;
}



void TAppWindow::clear_display()
{
    SetPort(fDocWindow);
    EraseRect(&fDocWindow->portRect);
}

void TAppWindow::redraw_display()
{
    SetPort(fDocWindow);
    EraseRect(&fDocWindow->portRect);
    draw_update_contents();
}

void TAppWindow::do_about()
{
    Alert(128, nil);
}

void doNodeEdit(char *prompt, char *text);

void TAppWindow::do_edit(char * prompt, char *buf)
{
    doNodeEdit(prompt, buf);
}

void TAppWindow::show_info(char *buf)
{
    do_edit(" ", buf);
}


int TAppWindow::choose_one(char *prompt1, char *prompt2)
{
    DialogPtr dLog;
    short itemHit = 0;
    GrafPort *savePort;
    char buf1[257], buf2[257];
    Handle itemH;  Rect r;
    short itemType;
    GetPort (&savePort);

    dLog = GetNewDialog (515, (Ptr)0L, (WindowPtr) -1);

    InitCursor ();
    
    SetPort(dLog);
    
    itemHit = 0;
    
    // Set the prompt text field:

    GetDItem(dLog, (short)1, &itemType, &itemH, &r);
    if (itemH != (Handle) 0) {
        sprintf(&(buf1[0]),"%s",prompt1);
        unsigned char * pascal_buf1 = CtoPstr(buf1);
        SetCTitle((ControlHandle)itemH, pascal_buf1);
    }

    GetDItem(dLog, (short)2, &itemType, &itemH, &r);
    if (itemH != (Handle) 0) {
        sprintf(&(buf2[0]),"%s",prompt2);
        unsigned char * pascal_buf2 = CtoPstr(buf2);
        SetCTitle((ControlHandle)itemH, pascal_buf2);
    }

    while ((itemHit != 1) && (itemHit != 2))
                ModalDialog(nil,&itemHit);
                                    
    DisposDialog (dLog);
    SetPort (savePort);
    return itemHit - 1;  // Return 0 or 1, NOT 1 or 2
}

// Handle single list selection (like Figure 5.3):

static pascal void myItem(DialogPtr dlg, short itemNumber)
{
    Handle itemHandle;
    short itemType;
    Rect itemRect;
    GetDItem(dlg, itemNumber, &itemType, &itemHandle, &itemRect);
    if (itemNumber == 3) {  // list user item
        ListHandle myList =
            (ListHandle)((DialogPeek)dlg)->window.refCon;
        LUpdate(dlg->visRgn, myList);
        InsetRect(&itemRect, -2, -2);
        FrameRect(&itemRect);
    }
}

static int itemSelected;

pascal Boolean myFilter(DialogPtr dlg, EventRecord *event,
                        short *itemHit)
{
    itemSelected = -1;
    short itemType;
    Handle itemH;
    Rect itemR;
    if (event->what == mouseDown)
    {
        GetDItem(dlg, (short)3, &itemType,
                 &itemH, &itemR);  // list box
      Point where = event->where;
      GlobalToLocal(&where);
        if (PtInRect(where, &itemR))
        {
            ListHandle myList = 
                (ListHandle)((DialogPeek)dlg)->window.refCon;
            if (LClick(where, event->modifiers, myList))
            {
                Point cellSelected = LLastClick(myList);
                itemSelected = cellSelected.v;
                *itemHit = 1;
                return true;
            }
        }
        GetDItem(dlg, (short)1, &itemType,
                 &itemH, &itemR);  // Cancel button
        if (PtInRect(where, &itemR))
        {
             *itemHit = 1;
             return true; 
        }
        *itemHit = 3;
        return true;
    }
    if (event->what == keyDown)
    {
        *itemHit = 1;
        return true; 
    }
    return false;
}

int TAppWindow::choose_one_from_list(char *prompt,
                                     char **list_items,
                                     int number_of_items)
{
    DialogPtr dLog;
    short itemHit = 0;
    GrafPort *savePort;
    char buf[257];
    Handle itemH;  Rect r;
    short itemType;
    GetPort (&savePort);

    dLog = GetNewDialog (511, (Ptr)0L, (WindowPtr) -1);

    InitCursor ();

    // Set up data for List Manager:
    SetPort(dLog);
    Rect itemR, dataR;
    GetDItem(dLog, (short)3, &itemType, &itemH, &itemR);
    SetDItem(dLog, (short)3, itemType, (Handle)myItem, &itemR);
    
    itemR.right -=15;
    SetRect(&dataR, 0, 0, 1, number_of_items);
    Point cellSize; SetPt(&cellSize, 260, 18);
    ListHandle myList = LNew(&itemR,&dataR, cellSize, 0,
                             (WindowPtr)dLog, false, false,
                             true, true);
    // As per TN example, allow dialog access to List data structures:
    ((DialogPeek)dLog)->window.refCon = (long)myList;
    
    // Initialize the list cell contents:
    for (int i=0; i<number_of_items; i++) 
    {
        Point cellLocation;
        cellLocation.v = i; cellLocation.h = 0;
        LSetCell(list_items[i], strlen(list_items[i]),
                 cellLocation, myList);
    }
    LDoDraw(true,myList);
    
    itemHit = 0;
    SetRect(&r,0,0,400,400);
    
    // Set the prompt text field:
    GetDItem(dLog, (short)2, &itemType, &itemH, &r);
    sprintf(&(buf[0]),"%s",prompt);
    unsigned char * pascal_buf = CtoPstr(buf);
    SetIText(itemH, pascal_buf);
    
    while ((itemHit != 1))
        ModalDialog(myFilter,&itemHit);
                                    
    if (itemHit == 1) {  // Cancel

    }
    
    LDispose(myList);
    DisposDialog (dLog);
    SetPort (savePort);
    return itemSelected;
}


void TAppWindow::init_scrolling_text(int top, int right,
                                     int bottom, int left)
{
    top_text_clip    = top;
    right_text_clip  = right;
    bottom_text_clip = bottom;
    left_text_clip   = left;
    number_of_saved_lines =
         1 + ((bottom_text_clip - top_text_clip) /
                         string_height(" "));
    if (number_of_saved_lines > MAX_TEXT_LINES)
       number_of_saved_lines = MAX_TEXT_LINES;
    Rect r;         
    SetRect(&r,left_text_clip,top_text_clip-2,
            right_text_clip,bottom_text_clip+2);
    EraseRect(&r);

    current_ring_buffer_start = 0;
    // Allocate storage for saved text lines:
    for (int i=0; i<number_of_saved_lines; i++) 
    {
        saved_text[i] = new char[MAX_TEXT_LINE_SIZE + 1];
        saved_text[i][0] = '\0';
    }
    in_scrolling_text_mode = 1;
    redraw_both_text_and_graphics = 1;
    // Set the font to a non-proportional spacing font:
    TextFont(4);  //  Monoco (display looks better with a
                  // non-proportional font)
}

void TAppWindow::init_scrolling_text()
{   
    // Set the text area with content rectangle of the window:
    Rect r = fDocWindow->portRect;
    top_text_clip    = r.top+5;
    right_text_clip  = r.right+11;
    bottom_text_clip = r.bottom-4;
    left_text_clip   = r.left+5;
    number_of_saved_lines = 
             1 + ((bottom_text_clip - top_text_clip) /
                     string_height(" "));
    if (number_of_saved_lines > MAX_TEXT_LINES)
       number_of_saved_lines = MAX_TEXT_LINES;
    Rect r2;
    SetRect(&r2,left_text_clip,top_text_clip-2,
            right_text_clip,bottom_text_clip+2);
    EraseRect(&r2);
    
    current_ring_buffer_start = 0;
    // Allocate storage for saved text lines:
    for (int i=0; i<number_of_saved_lines; i++) {
        saved_text[i] = new char[MAX_TEXT_LINE_SIZE + 1];
        saved_text[i][0] = '\0';
    }
    in_scrolling_text_mode = 1;
    redraw_both_text_and_graphics = 0; // entire window used
                                       // for scrolling text.
    // Set the font to a non-proportional spacing font:
    TextFont(4);  //  Monoco (display looks better with a
                  //  non-proportional font)
}

void TAppWindow::put_scrolling_text(char *str)
{
    if (bottom_text_clip != 0)  {
        Rect saveVis= fDocWindow->portRect;
        Rect r;
        RgnHandle h = NewRgn();
        SetRect(&r,left_text_clip,top_text_clip,
                right_text_clip,bottom_text_clip);
        ClipRect(&r);
        ScrollRect(&r, 0, -string_height(str) - 2, h);
        plot_string(left_text_clip + 1, 
                    bottom_text_clip - 2, str);
        DisposeRgn(h);
        char buf[257];
        if (strlen(str) < 256) 
        {
            sprintf(buf,"%s",str);
            buf[MAX_TEXT_LINE_SIZE] = '\0';
            sprintf(saved_text[current_ring_buffer_start],
                    "%s",buf);
            current_ring_buffer_start++;
            if (current_ring_buffer_start > 
                (number_of_saved_lines - 1))
            {
                current_ring_buffer_start = 0;
            }
        }
        ClipRect(&saveVis);
        if (redraw_both_text_and_graphics == 1) 
        {
            r.bottom += 2; r.top -= 2;
            FrameRect(&r);
        }
    } else {
        Warning("TAppWindow: called put_scrolling_text before init_scrolling_text");
    }
    in_scrolling_text_mode = 1;
}


void TAppWindow::reset_scrolling_text()
{
    for (int i=0; i<MAX_TEXT_LINES; i++)
    {
        saved_text[i][0] = '\0';
    }
}


static char *fileExtension;

pascal Boolean fileFilter(ParmBlkPtr paramBlock) 
{
    int size2 = strlen(fileExtension);
    StringPtr sPtr = paramBlock->fileParam.ioNamePtr;
    char * fileName = (char *)sPtr;
    int size = fileName[0];
    char buf[257];
    for (int i=0; i<size; i++) buf[i] = fileName[i+1];
    buf[size] = '\0';
    // Find the position of the first '.' character of
    // the extension (if any):
    if (size2 > 0)
    {
        for (i=0; i<size; i++)
            if (buf[i] == '.')
            {
                if ((size - i) >= size2)
                {
                    int found_match = 1;
                    for (int j=i+1; j<size; j++)
                        if (buf[j] != fileExtension[j - i - 1]) 
                            found_match = 0;
                    if (found_match)
                    {
                        return false;
                    }
                }
            }
        return true; // skip this file name
    }
    return false; // no file extension specified,
                  // so return every file
}

int TAppWindow::choose_file_to_read(char * prompt,
                                    char * extension, 
                                    char *returned_filename)
            // 0 return -> OK, 1 return -> Failure
{
    fileExtension = extension;
    SFReply sf;  SFTypeList tl; 
    Point p; p.h = 100; p.v = 100;
    char prompt_buf[256];
    sprintf(prompt_buf,"%s", prompt);
    unsigned char * pascal_prompt_buf = CtoPstr(prompt_buf);
    SFGetFile(p, pascal_prompt_buf,
              (FileFilterProcPtr)fileFilter, -1,
              tl, (DlgHookProcPtr)-1L, &sf);
    
    int size = sf.fName[0];
    if (sf.good == 0)  size = 0;
    for (int i=0; i<size; i++)
       returned_filename[i] = sf.fName[i+1];
    returned_filename[size] = '\0';
    if (size > 0)   return 0;
     else           return 1;
}

int TAppWindow::choose_file_to_write(char *prompt,
                                     char *returned_filename)
          // 0 return -> OK, 1 return -> Failure
{
    SFReply sf;
    Point p; p.h = 100; p.v = 100;
    char prompt_buf[256];
    sprintf(prompt_buf, "%s", prompt);
    unsigned char * pascal_prompt_buf = CtoPstr(prompt_buf);
    SFPutFile(p, pascal_prompt_buf, "\p",
              (DlgHookProcPtr)nil, &sf);
    int size = sf.fName[0];
    for (int i=0; i<size; i++)
        returned_filename[i] = sf.fName[i+1];
    returned_filename[size] = '\0';
    if (size > 0)   return 0;
     else           return 1;
}

void Warning(char *message)
{
    char buf[255];
    sprintf(buf," %s",message);
    unsigned char * pascal_buf = CtoPstr(buf);

    SetCursor(&arrow);
    ParamText(pascal_buf, "\p", "\p", "\p");
    (void) Alert(129, (ModalFilterProcPtr) nil);
              // 129 is the ALRT resource number
    current_window->show_info(message);
}



#define NAME_SIZE 256
#define TEXT_SIZE 256

void doNodeEdit(char *prompt, char *text)
{
    DialogPtr dLog;
    short itemHit = 0;
    GrafPort *savePort;
    char buf[257];
    /*int itemType;*/  Handle itemH;  Rect r; int len;
    short itemType;
    GetPort (&savePort);

    dLog = GetNewDialog (501, (Ptr)0L, (WindowPtr) -1);

    InitCursor ();

    itemHit = 0;
    SetRect(&r,0,0,400,400);

    GetDItem(dLog, (short)4, &itemType, &itemH, &r);
    sprintf(&(buf[0]),"%s",prompt);
    unsigned char * pascal_buf = CtoPstr(buf);
    SetIText(itemH, pascal_buf);
    
    GetDItem(dLog, (short)3, &itemType, &itemH, &r);
    sprintf(&(buf[0]),"%s",text);
    pascal_buf = CtoPstr(buf);
    SetIText(itemH, pascal_buf);
    
    while ((itemHit != 2) && (itemHit != 1))
                ModalDialog(nil,&itemHit);
                                    
    if (itemHit == 1) {  // OK
        GetDItem(dLog, (short)3, &itemType, &itemH, &r);
        unsigned char * pascal_buf = CtoPstr(buf);
        GetIText(itemH, pascal_buf);
        char * buf = PtoCstr(pascal_buf);
        if (TEXT_SIZE>strlen(buf)) len = strlen(buf);  
        else len = TEXT_SIZE - 1;
        for (int i=0; i<len; i++) text[i] = buf[i];
        text[len] = '\0';
    }
    
    DisposDialog (dLog);
    SetPort (savePort);
}

void setup_menus(char *menu_name, int num_items, char ** items)

{
    MenuHandle  appMenu;

    char buf[64];
    sprintf(buf,"%s", menu_name);
    unsigned char *pascal_cp = CtoPstr(buf); // copies in place
    
    InsertMenu(appMenu  = NewMenu(mApp, pascal_cp), 0);
    DrawMenuBar();
    for (int i=0; i<num_items; i++) {
        sprintf(buf,"%s", items[i]);
        pascal_cp = CtoPstr(buf); // copies in place
        AppendMenu(appMenu, pascal_cp);
    }
}


char Application::prompt_for_dialog[
                   Application::DIALOG_TEXT_LENGTH];
char Application::file_extension_for_dialog[
                   Application::DIALOG_TEXT_LENGTH];
char Application::text_from_dialog[
                   Application::DIALOG_TEXT_LENGTH];
char * Application::list_of_items_for_dialog[
                   Application::DIALOG_MAX_LIST_ITEMS];
int Application::number_of_dialog_list_items;
TAppWindow * Application::fWindow = (TAppWindow *)NULL;

